--  FILE:    graphen.adb
--  PROJECT: Programmieruebungen, Uebungsblatt 11
--  VERSION: 1.0
--  DATE:    27.01.2007
--  AUTHOR:  http://CodeWelt.com
--
-------------------------------------------------------------------
-- 
--  Aufgabe 11.2: Externe Reprsentation
--
--  In dieser Aufgabe wird ein gerichteter Graph aus der Datei
--  graphen.txt geladen und mit Hilfe von Adjazenzlisten
--  reprsentiert. Aus dieser Darstellung wird eine
--  Adjazenzmatrix des Graphen erstellt.
--  Die graphen.txt ist wie folgt aufgebaut: In jeder Zeile
--  der Datei ist eine Kante spezifiziert mit Startknoten-Id,
--  Kantengewicht und Zielknoten-Id.
--  Eine Zeile hat folgendes Format, wobei Source fr die
--  Knoten-Id des Quellknotens steht, Target fr die Knoten-Id
--  des Zielknotens und W fr das Gewicht der Kante:
--  Source (W) Target
--
-------------------------------------------------------------------
with Ada.Text_IO, Ada.Strings.Unbounded, Ada.Strings.Unbounded.Text_IO,
     Ada.Strings.Fixed, Ada.Unchecked_Deallocation;
use  Ada.Text_IO, Ada.Strings.Unbounded, Ada.Strings.Unbounded.Text_IO;

package body graphen is

   --  Zwei Instanzen von Ada.Unchecked_Deallocation fr die zwei
   --  unterschiedlichen Listen Knotenliste und Kantenliste.
   procedure FreeKnoten is new Ada.Unchecked_Deallocation
   (Knoten, Knotenliste);
   procedure FreeKante is new Ada.Unchecked_Deallocation (Kante, Kantenliste);

   --  Der Anchor zur Knotenliste als globale Variable.
   KnotenlisteAnchor : Knotenliste := null;
   --  Es wird beim Hinzufgen von Elementen in die Knotenliste
   --  die Anzahl der Elemente in der Knotenliste gezhlt.
   KnotenlisteCounter : Integer := 0;
   --  Unter allen hinzugefgten Knoten wird das Element gesucht
   --  mit der Lngsten Id um spter die Matrix formatiert
   --  auszugeben.
   LaengsteId : Integer := 0;


   --  FUNCTION Is_Empty
   --
   --  Es wird ermittelt ob der bergebene Zeiger
   --  auf eine Knotenliste leer ist oder nicht.
   --
   --  PARAMETERS:
   --  + Anchor - Zeiger auf eine Knotenliste.
   --  RETURNS:
   --  Die Funktion liefert True zurck wenn der
   --  bergebene Zeiger null ist, andernfalls
   --  False.
   function Is_Empty
     (Anchor : in Knotenliste)
     return Boolean
   is
   begin
      return Anchor = null;
   end Is_Empty;


   --  FUNCTION Zerlege
   --
   --  Beim Einlesen der Datei wird
   --  jede Zeile in ihre Teile zerlegt.
   --  Um nacher mit den daraus gewonnenen Daten
   --  weiterarbeiten zu knnen.
   --
   --  PARAMETERS:
   --  + Line - Die aktuelle Zeile aus der Datei
   --  als Unbounded_String welche in Knoten-Id des
   --  Quellknotens, Gewicht der Kante und Knoten-Id
   --  des Zielknotens zerlegt werden soll.
   --  RETURNS:
   --  Die Funktion liefert den in der graphen.ads
   --  definierten Typ SourceWTarget zurck. Dieser
   --  Typ enthlt die gewonnenen Daten.
   function Zerlege
      (Line : in Unbounded_String)
      return SourceWTarget
   is
      KlammerAufPos, SpacePos, KlammerZuPos : Integer := 0;
      Zerlegt : SourceWTarget;
   begin
      --  Die Positionen der Klammeren und der Leerzeile werden
      --  bestimmt.
      KlammerAufPos := Ada.Strings.Fixed.Index (To_String (Line), "(",
      Ada.Strings.Forward);
      KlammerZuPos := Ada.Strings.Fixed.Index (To_String (Line), ")",
      Ada.Strings.Forward);
      SpacePos := Ada.Strings.Fixed.Index (To_String (Line), " ",
      Ada.Strings.Forward);
      --  Die aus der bergebenen Zeile ermittelten Daten werden
      --  in den Typ SourceWTarget hinterlegt und zurckgegeben.
      Zerlegt.W := Natural'Value (Slice (Line, KlammerAufPos + 1,
      KlammerZuPos - 1));
      Zerlegt.Target := To_Unbounded_String (Slice (Line, KlammerZuPos + 2,
      Length (Line)));
      Zerlegt.Source := To_Unbounded_String (Slice (Line, 1, SpacePos - 1));
      return Zerlegt;
   end Zerlege;


   --  FUNCTION In_Knotenliste
   --
   --  Beim Einlesen der Datei wird
   --  gefragt ob der aktuell betrachtete Knoten
   --  bereits in der Liste der Knoten vorhanden
   --  ist oder nicht.
   --
   --  PARAMETERS:
   --  + Idx - Die Id des aktuell betrachteten
   --  Knotens als Unbounded_String zum
   --  Identifizieren und Vergleichen.
   --  RETURNS:
   --  Die Funktion liefert True zurck wenn
   --  die bergebene Knoten-Id bereits in der
   --  Liste vorhanden ist, andernfalls False.
   function In_Knotenliste
      (Idx : in Unbounded_String)
      return Boolean
   is
      Check : Knotenliste := KnotenlisteAnchor;
   begin
      if Is_Empty (KnotenlisteAnchor) = True then
         return False;
      end if;
      Check := KnotenlisteAnchor;
      --  Die Schleife luft durch alle Elemente der
      --  Knotenliste.
      while Check /= null loop
         --  Wenn die Id des Elements gleich ist wie
         --  der bergebene Unbounded_String, welcher
         --  gesucht wird, wird True zurckgegeben.
         if Check.Id = Idx then
            return True;
         end if;
         Check := Check.Next;
      end loop;
      --  Wenn keine bereinstimmung gefunden wurde,
      --  ist die gesuchte Id nicht in der Liste
      --  vorhanden und es wird False zurckgegeben.
      return False;
   end In_Knotenliste;
   

   --  PROCEDURE AddToKnotenliste
   --
   --  Es wird ein neues Element mit der
   --  als Unbounded_String bergebenen Id 
   --  an die Knotenliste angefgt.
   --
   --  PARAMETERS:
   --  + New_Id - Die Id des neuen Elements.
   procedure AddToKnotenliste
     (New_Id : in Unbounded_String)
   is
      Last : Knotenliste := KnotenlisteAnchor;
   begin
      --  Unter allen hinzugefgten Knoten wird das Element gesucht
      --  mit der Lngsten Id um spter die Matrix formatiert
      --  auszugeben.
      if Length (New_Id) > LaengsteId then
         LaengsteId := Length (New_Id);
      end if;
      --  Es wird beim Hinzufgen von Elementen in die Knotenliste
      --  die Anzahl der Elemente in der Knotenliste gezhlt.
      KnotenlisteCounter := KnotenlisteCounter + 1;
      if Is_Empty (KnotenlisteAnchor) then
         --  falls zuvor leere Liste, dann einzellige Liste erzeugen
         KnotenlisteAnchor := new Knoten'(Next => null, Kanten => null,
         Id => New_Id);
      else
         --  letzte Zelle suchen
         Last := KnotenlisteAnchor;
         while Last.Next /= null loop
            Last := Last.Next;
         end loop;
         --  ... und Next der letzten Zelle auf eine neue Zelle setzen
         Last.Next := new Knoten'(Next => null, Kanten => null, Id => New_Id);
      end if;
   end AddToKnotenliste;


   --  PROCEDURE Load
   --
   --  In dieser Prozedur wird die Datei graphen.txt gelesen
   --  und die daraus resultierenden Listen ermittelt.
   procedure Load is
      File : Ada.Text_IO.File_Type;
      AktuelleZeile : Unbounded_String := Null_Unbounded_String;
      Zerlegt : SourceWTarget;
      Abbruch : Boolean := False;
      KnotenlisteAnchorTemp, KnotenlisteAnchorTemp2 : Knotenliste
      := KnotenlisteAnchor;
   begin
      Open (File, in_file, "graphen.txt");
      --  Es folgen zwei while Schleifen die jede Zeile der
      --  graphen.txt durchlaufen. Das erste Mal werden
      --  die Knoten festgelegt und beim zweiten Mal die
      --  Kanten zu jedem Knoten bestimmt.
      while not End_Of_File (File) loop
         Get_Line (File, AktuelleZeile);
         Zerlegt := Zerlege (AktuelleZeile);
         --  Es werden in dieser while Schleife nur die
         --  Knoten festgelegt. Es wird daher Source
         --  und Target als mglicher Knoten betrachtet.
         --  Nur wenn die aktuell betrachtete Knoten-Id
         --  noch nicht in der Knotenliste vorhanden ist
         --  wird ein neuer Knoten an die Liste angehngt.
         if In_Knotenliste (Zerlegt.Source) = False then
            AddToKnotenliste (Zerlegt.Source);
         end if;         
         if In_Knotenliste (Zerlegt.Target) = False then
            AddToKnotenliste (Zerlegt.Target);
         end if;
      end loop;
      --  Nach dem ersten Durchlaufen der Datei wird
      --  sie zurckgesetzt um fr die folgende while
      --  Schleife neu zu beginnen.
      Reset (File);
      AktuelleZeile := Null_Unbounded_String;
      while not End_Of_File (File) loop
         Get_Line (File, AktuelleZeile);
         Zerlegt := Zerlege (AktuelleZeile);
         --  Es werden zwei Zeiger auf die Knotenliste
         --  gebraucht um erstens den Quellknoten
         --  und zweitens den Zielknoten festzuhalten.
         KnotenlisteAnchorTemp := KnotenlisteAnchor;
         KnotenlisteAnchorTemp2 := KnotenlisteAnchor;
         Abbruch := False;
         --  Die Schleife luft solange bis der Zeiger
         --  KnotenlisteAnchorTemp am Ende der Liste angekommen ist.
         while Is_Empty (KnotenlisteAnchorTemp) /= True loop
            --  Wenn der aktuell betrachtete Knoten der gesuchte
            --  Quellknoten der Kante ist wird fortgefahren.
            if KnotenlisteAnchorTemp.Id = Zerlegt.Source then
               --  Die Schleife luft solange bis der Zeiger
               --  KnotenlisteAnchorTemp2 am Ende der Liste angekommen ist.
               --  Es werden alle Elemente der Liste durchlaufen bis
               --  die gesuchte Knoten-Id gefunden wurde.             
               while Is_Empty (KnotenlisteAnchorTemp2) /= True loop
                  --  Wenn die aktuelle Knoten-Id dem gesuchten Zielknoten
                  --  entspricht ...
                  if KnotenlisteAnchorTemp2.Id = Zerlegt.Target then
                     --  ... wird ein neues Kanten-Element an die Liste
                     --  der Kanten des zuvor gesuchten Elements der
                     --  Knotenliste angehngt.
                     --  Das neue Element der Kantenliste wird
                     --  das zuvor ermittelte Gewicht und der Zielknoten
                     --  welcher in der obigen while Schleife gefunden wurde
                     --  zugeordnet.
                     --  falls zuvor leere Liste,
                     --  dann einzellige Liste erzeugen
                     if KnotenlisteAnchorTemp.Kanten = null then
                        KnotenlisteAnchorTemp.Kanten := new Kante'(null,
                        KnotenlisteAnchorTemp2, Zerlegt.W);
                        Abbruch := True;
                        exit;
                     else
                        --  Neues Element in die Kantenliste anfgen.
                        KnotenlisteAnchorTemp.Kanten :=
                        new Kante'(KnotenlisteAnchorTemp.Kanten,
                        KnotenlisteAnchorTemp2, Zerlegt.W);
                        Abbruch := True;
                        exit;
                     end if;
                  end if;
                  KnotenlisteAnchorTemp2 := KnotenlisteAnchorTemp2.Next;
               end loop;
            end if;
            if Abbruch = True then
               exit;
            end if;
            KnotenlisteAnchorTemp := KnotenlisteAnchorTemp.Next;
         end loop;
      end loop;
      Close (File);
   end Load;
   

   --  FUNCTION Get_Koord
   --
   --  Zur Ausgabe wird eine Matrix erzeugt. Beim
   --  Setzen der Gewichte in die Matrix wird
   --  eine horizontale Koordinate gebraucht.
   --
   --  PARAMETERS:
   --  + SearchUnb - Der gesuchte Knoten-Id dessen
   --  horizontale Koordinate ermittelt werden soll.
   --  RETURNS:
   --  Die Funktion liefert einen Integer-Wert
   --  zurck der die Position des gesuchten
   --  Knoten-Ids in der Knotenliste reprsentiert.
   function Get_Koord
      (SearchUnb : Unbounded_String)
      return Integer
   is
      KnotList : Knotenliste := KnotenlisteAnchor;
      ReturnMe : Integer := 0;
   begin
      while Is_Empty (KnotList) /= True loop 
         ReturnMe := ReturnMe + 1;
         if KnotList.Id = SearchUnb then
            return ReturnMe;            
         end if;
         KnotList := KnotList.Next;
      end loop;
      return 0;
   end Get_Koord;
   

   --  FUNCTION Fill_with_Spaces
   --
   --  Zur formatierten Ausgabe wurde oben die
   --  lngste Id ermittelt. Diese Funktion
   --  fllt eine gegebene Zeichenkette solange
   --  mit Leerzeichen bis sie die selbe
   --  lnge wie die lngste Id hat. 
   --
   --  PARAMETERS:
   --  + FillMe - Die gegebene Zeichenkette
   --  dessen lnge angeglichen werden soll.
   --  RETURNS:
   --  Die Funktion liefert einen Unbounded_String
   --  zurck mit angeglichener Lnge, gefllt
   --  mit Leerzeichen.
   function Fill_with_Spaces
      (FillMe : Unbounded_String)
      return Unbounded_String
   is
      ReturnMe : Unbounded_String := FillMe;
   begin
      while Length (ReturnMe) < (LaengsteId + 1) loop
         ReturnMe := " " & ReturnMe;
      end loop;
      return ReturnMe;
   end Fill_with_Spaces;


   procedure Destroy is
      KnotenlisteAnchorx, KnotenlisteAnchorxCopy : Knotenliste
      := KnotenlisteAnchor;
      KantenlisteAnchorx, KantenlisteAnchorxCopy : Kantenliste := null;
   begin
      --  Die Schleife luft fr jedes Element der Knotenliste.
      while Is_Empty (KnotenlisteAnchorx) /= True loop
         KantenlisteAnchorx := KnotenlisteAnchorx.Kanten;
         --  Die Schleife luft fr jedes Element der Kantenliste
         --  des aktuellen Elements der Knotenliste.
         while KantenlisteAnchorx /= null loop
            KantenlisteAnchorxCopy := KantenlisteAnchorx;
            KantenlisteAnchorx := KantenlisteAnchorx.Next;
            FreeKante (KantenlisteAnchorxCopy);
         end loop;
         KnotenlisteAnchorxCopy := KnotenlisteAnchorx;
         KnotenlisteAnchorx := KnotenlisteAnchorx.Next;
         FreeKnoten (KnotenlisteAnchorxCopy);
      end loop;
      KnotenlisteAnchor := KnotenlisteAnchorx;
   end Destroy;


   --  PROCEDURE Ausgabe
   --
   --  In dieser Prozedur wird eine Matrix in den Dimensionen,
   --  bestimmt durch die Anzahl der Elemente in der Knotenliste,
   --  definiert, mit den Kantengewichten gesetzt und formatiert
   --  ausgegeben. Es werden alle Felder der Matrix fr den Anfang
   --  auf null gesetzt.
   procedure Ausgabe is
      Matrix : array (1 .. KnotenlisteCounter, 1 .. KnotenlisteCounter) of 
      Integer := (others => (others => 0));
      KnotenlisteAnchorx : Knotenliste := KnotenlisteAnchor;
      KantenlisteAnchorx : Kantenliste := null;
      VerticalCounter : Integer := 1;
   begin
      --  Es folgen zwei while Schleifen. In der Ersten
      --  wird durch alle Kanten jedes Knotens der
      --  Knotenliste gelaufen und die Kantengewichtige
      --  in die Matrix bertragen. In der Zweiten Schleife
      --  wird die Matrix formatiert ausgegeben.
      Put (Fill_with_Spaces (Null_Unbounded_String));
      while Is_Empty (KnotenlisteAnchorx) /= True loop
         Put (Fill_with_Spaces (KnotenlisteAnchorx.Id));
         KantenlisteAnchorx := KnotenlisteAnchorx.Kanten;
         while KantenlisteAnchorx /= null loop
            Matrix (Get_Koord (KantenlisteAnchorx.Kantenziel.Id),
            VerticalCounter) := KantenlisteAnchorx.Gewicht;
            KantenlisteAnchorx := KantenlisteAnchorx.Next;
         end loop;
         KnotenlisteAnchorx := KnotenlisteAnchorx.Next;
         VerticalCounter := VerticalCounter + 1;
      end loop;
      New_Line;
      KnotenlisteAnchorx := KnotenlisteAnchor;
      VerticalCounter := 1;
      while Is_Empty (KnotenlisteAnchorx) /= True loop
         Put (Fill_with_Spaces (KnotenlisteAnchorx.Id));
         for Laufvar in 1 .. KnotenlisteCounter loop
            Put (Fill_with_Spaces (To_Unbounded_String (Matrix (Laufvar,
            VerticalCounter)'Img)));
         end loop;
         New_Line;
         KnotenlisteAnchorx := KnotenlisteAnchorx.Next;
         VerticalCounter := VerticalCounter + 1;
      end loop;
      --  Nachdem ausgegeben wurde, wird der bentigte Speicher freigegeben.
      Destroy;
   end Ausgabe;

end graphen;
